# AntURL - Professional URL Shortener

A modern, secure, and professional URL shortener built with PHP and MySQL.

## 🚀 Features

- **Fast URL Shortening**: Generate 6-character unique short codes
- **Smart Duplicate Detection**: Returns existing short URL if the same long URL already exists
- **Secure**: Uses prepared statements to prevent SQL injection
- **Responsive Design**: Clean, modern UI that works on all devices
- **Error Handling**: Proper 404 pages and error messages
- **URL Validation**: Validates URLs before storing
- **Copy to Clipboard**: Easy one-click copying of short URLs
- **SEO Friendly**: Proper HTTP status codes and redirects

## 📁 File Structure

```
url/
├── index.php          # Main page with URL shortening form
├── redirect.php       # Handles URL redirection
├── db.php            # Database connection and operations
├── style.css         # Modern CSS styling
├── database.sql      # Database schema
├── .htaccess         # Apache URL rewriting rules
└── README.md         # This file
```

## 🛠️ Installation

### Prerequisites

- PHP 7.4 or higher
- MySQL 5.7 or higher
- Apache/Nginx web server
- mod_rewrite enabled (for Apache)

### Setup Steps

1. **Database Setup**
   ```sql
   CREATE DATABASE url_shortener CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
   ```
   
   Then run the SQL commands from `database.sql`:
   ```bash
   mysql -u root -p url_shortener < database.sql
   ```

2. **Configure Database Connection**
   
   Edit the database credentials in `db.php`:
   ```php
   private $host = 'localhost';
   private $dbname = 'url_shortener';
   private $username = 'your_username';  // Change this
   private $password = 'your_password';  // Change this
   ```

3. **Web Server Configuration**

   **For Apache:**
   - Ensure `mod_rewrite` is enabled
   - The included `.htaccess` file will handle URL rewriting
   - Point your domain to the project directory

   **For Nginx:**
   ```nginx
   server {
       listen 80;
       server_name anturl.com www.anturl.com;
       root /path/to/your/project;
       index index.php;

       location / {
           try_files $uri $uri/ /redirect.php$is_args$args;
       }

       location ~ \.php$ {
           fastcgi_pass unix:/var/run/php/php7.4-fpm.sock;
           fastcgi_index index.php;
           include fastcgi_params;
           fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
       }
   }
   ```

4. **Domain Configuration**
   
   Update the domain in `index.php`:
   ```php
   private $domain = 'https://anturl.com';  // Change to your domain
   ```

## 🎯 Usage

1. **Shorten a URL:**
   - Visit your domain (e.g., https://anturl.com)
   - Enter a long URL in the input field
   - Click "Shorten URL"
   - Copy the generated short URL

2. **Access a short URL:**
   - Visit the short URL (e.g., https://anturl.com/abc123)
   - You'll be automatically redirected to the original URL

## 🔧 Features Explained

### URL Validation
- Automatically adds `http://` if no protocol is specified
- Uses PHP's `filter_var()` for proper URL validation

### Short Code Generation
- Generates 6-character codes using letters (a-z, A-Z) and numbers (0-9)
- Automatically checks for uniqueness before inserting
- Regenerates if collision occurs

### Security
- All database queries use prepared statements
- Input sanitization and validation
- XSS protection with `htmlspecialchars()`

### Error Handling
- Custom 404 page for invalid short codes
- Database connection error handling
- User-friendly error messages

## 🎨 Customization

### Styling
The CSS uses CSS custom properties (variables) for easy theming:
```css
:root {
    --primary-gradient: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    --success-color: #10b981;
    --error-color: #ef4444;
    /* ... more variables */
}
```

### Short Code Length
Change the length in `index.php`:
```php
private function generateShortCode($length = 6) // Change 6 to desired length
```

### Domain
Update the domain in the `UrlShortener` class:
```php
private $domain = 'https://yourdomain.com';
```

## 🚦 Testing

1. **Test URL Shortening:**
   - Enter various URLs (with/without http://)
   - Verify short URLs are generated
   - Test duplicate URL detection

2. **Test Redirection:**
   - Visit generated short URLs
   - Verify redirection works
   - Test invalid short codes show 404

3. **Test Error Handling:**
   - Enter invalid URLs
   - Test with database disconnected
   - Verify error messages display properly

## 📊 Database Schema

```sql
CREATE TABLE urls (
    id INT AUTO_INCREMENT PRIMARY KEY,
    long_url TEXT NOT NULL,
    short_code VARCHAR(10) NOT NULL UNIQUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_short_code (short_code),
    INDEX idx_created_at (created_at)
);
```

## 🔒 Security Considerations

1. **Rate Limiting**: Consider implementing rate limiting for production
2. **HTTPS**: Always use HTTPS in production
3. **Input Validation**: All inputs are validated and sanitized
4. **SQL Injection**: Prevented using prepared statements
5. **XSS**: Prevented using proper output encoding

## 🌟 Production Deployment

1. **Environment Variables**: Store database credentials in environment variables
2. **Error Logging**: Enable PHP error logging
3. **Monitoring**: Set up monitoring for uptime and performance
4. **Backups**: Regular database backups
5. **CDN**: Consider using a CDN for static assets

## 📈 Performance Tips

1. **Database Indexing**: Indexes are created on `short_code` and `created_at`
2. **Connection Pooling**: Use persistent database connections for high traffic
3. **Caching**: Consider implementing Redis/Memcached for frequently accessed URLs
4. **Compression**: Enable gzip compression in web server

## 🤝 Contributing

Feel free to submit issues and enhancement requests!

## 📄 License

This project is open source and available under the [MIT License](LICENSE).

---

**Built with ❤️ for the web community**