<?php
require_once 'db.php';

class UrlShortener {
    private $db;
    private $domain;
    
    public function __construct() {
        $this->db = new Database();
        // Use current domain instead of hardcoded
        $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
        $this->domain = $protocol . '://' . $_SERVER['HTTP_HOST'];
    }
    
    public function shortenUrl($longUrl) {
        // Clean and normalize the URL
        $longUrl = trim($longUrl);
        
        // Add protocol if missing
        if (!preg_match('/^https?:\/\//', $longUrl)) {
            $longUrl = 'http://' . $longUrl;
        }
        
        // Validate URL
        if (!$this->isValidUrl($longUrl)) {
            return ['success' => false, 'message' => 'Please enter a valid URL'];
        }
        
        // Check if URL already exists
        $existing = $this->db->findByLongUrl($longUrl);
        if ($existing) {
            return [
                'success' => true,
                'short_url' => $this->domain . '/' . $existing['short_code'],
                'message' => 'URL already exists, here\'s your existing short link'
            ];
        }
        
        // Generate unique short code
        $shortCode = $this->generateShortCode();
        while ($this->db->shortCodeExists($shortCode)) {
            $shortCode = $this->generateShortCode();
        }
        
        // Insert into database
        if ($this->db->insertUrl($longUrl, $shortCode)) {
            return [
                'success' => true,
                'short_url' => $this->domain . '/' . $shortCode,
                'message' => 'Short URL created successfully!'
            ];
        }
        
        return ['success' => false, 'message' => 'Failed to create short URL'];
    }
    
    private function isValidUrl($url) {
        // Clean the URL first
        $url = trim($url);
        
        // Add http:// if no protocol specified
        if (!preg_match('/^https?:\/\//', $url)) {
            $url = 'http://' . $url;
        }
        
        return filter_var($url, FILTER_VALIDATE_URL) !== false;
    }
    
    private function generateShortCode($length = 6) {
        $characters = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
        $shortCode = '';
        for ($i = 0; $i < $length; $i++) {
            $shortCode .= $characters[random_int(0, strlen($characters) - 1)];
        }
        return $shortCode;
    }
}

$message = '';
$shortUrl = '';
$messageType = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && !empty($_POST['long_url'])) {
    $urlShortener = new UrlShortener();
    $result = $urlShortener->shortenUrl($_POST['long_url']);
    
    $message = $result['message'];
    $messageType = $result['success'] ? 'success' : 'error';
    if ($result['success']) {
        $shortUrl = $result['short_url'];
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>SpidyUrl - Free URL Shortener | Professional Link Management</title>
    <meta name="description" content="SpidyUrl is a free, professional URL shortener that creates clean, short links from long URLs. Fast, secure, and mobile-friendly with no registration required.">
    <meta name="keywords" content="URL shortener, link shortener, short URL, tiny URL, free URL shortener, link management, SpidyUrl">
    <meta name="author" content="SpidyUrl">
    <meta name="robots" content="index, follow">
    <link rel="canonical" href="https://spidyurl.com/">
    
    <!-- Open Graph / Facebook -->
    <meta property="og:type" content="website">
    <meta property="og:url" content="https://spidyurl.com/">
    <meta property="og:title" content="SpidyUrl - Free Professional URL Shortener">
    <meta property="og:description" content="Create short, clean URLs instantly with SpidyUrl. Fast, secure, and mobile-friendly URL shortening service.">
    <meta property="og:image" content="https://spidyurl.com/images/spider-logo-og.png">
    <meta property="og:site_name" content="SpidyUrl">
    
    <!-- Twitter -->
    <meta property="twitter:card" content="summary_large_image">
    <meta property="twitter:url" content="https://spidyurl.com/">
    <meta property="twitter:title" content="SpidyUrl - Free Professional URL Shortener">
    <meta property="twitter:description" content="Create short, clean URLs instantly with SpidyUrl. Fast, secure, and mobile-friendly URL shortening service.">
    <meta property="twitter:image" content="https://spidyurl.com/images/spider-logo-twitter.png">
    
    <!-- Favicon -->
    <link rel="icon" type="image/x-icon" href="/favicon.ico">
    <link rel="apple-touch-icon" sizes="180x180" href="/apple-touch-icon.png">
    
    <!-- Preconnect to external domains for performance -->
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    
    <link rel="stylesheet" href="style.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    
    <!-- Structured Data for SEO -->
    <script type="application/ld+json">
    {
        "@context": "https://schema.org",
        "@type": "WebApplication",
        "name": "SpidyUrl",
        "description": "Professional URL shortening service that creates clean, short links from long URLs",
        "url": "https://spidyurl.com",
        "applicationCategory": "Utility",
        "operatingSystem": "Web Browser",
        "offers": {
            "@type": "Offer",
            "price": "0",
            "priceCurrency": "USD"
        },
        "author": {
            "@type": "Organization",
            "name": "SpidyUrl",
            "url": "https://spidyurl.com"
        },
        "provider": {
            "@type": "Organization",
            "name": "SpidyUrl",
            "url": "https://spidyurl.com"
        },
        "featureList": [
            "Free URL shortening",
            "No registration required",
            "Duplicate URL detection",
            "Mobile-friendly interface",
            "Secure HTTPS redirects",
            "Professional short links"
        ],
        "screenshot": "https://spidyurl.com/images/screenshot.png",
        "softwareVersion": "2.0",
        "datePublished": "2024-09-18",
        "dateModified": "2024-09-18"
    }
    </script>
    
    <!-- Additional structured data for Organization -->
    <script type="application/ld+json">
    {
        "@context": "https://schema.org",
        "@type": "Organization",
        "name": "SpidyUrl",
        "url": "https://spidyurl.com",
        "logo": "https://spidyurl.com/images/spider-logo.png",
        "description": "Professional URL shortening service for creating clean, manageable short links",
        "contactPoint": {
            "@type": "ContactPoint",
            "email": "contact@spidyurl.com",
            "contactType": "customer service"
        },
        "sameAs": [
            "https://twitter.com/spidyurl",
            "https://linkedin.com/company/spidyurl"
        ]
    }
    </script>
</head>
<body>
    <div class="container">
        <h1 class="main-title">SPIDYURL</h1>
        
        <div class="main-card">
            <form method="POST" action="">
                <div class="section-header">
                    <span class="section-icon">🔗</span>
                    <span>Shorten a long URL</span>
                </div>
                
                <input 
                    type="text" 
                    name="long_url" 
                    class="url-input"
                    placeholder="Enter long link here" 
                    value="<?php echo htmlspecialchars($_POST['long_url'] ?? ''); ?>"
                    required
                >
                
                <button type="submit" class="shorten-btn">Shorten URL</button>
            </form>
            
            <?php if ($message): ?>
                <div class="message <?php echo $messageType; ?>">
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>
            
            <?php if ($shortUrl): ?>
                <div class="result">
                    <label class="result-label">Your shortened URL:</label>
                    <div class="url-result">
                        <input type="text" id="shortUrl" class="result-input" value="<?php echo htmlspecialchars($shortUrl); ?>" readonly>
                        <button onclick="copyToClipboard()" class="copy-btn">Copy</button>
                    </div>
                    <div class="share-options">
                        <button onclick="resetForm()" class="test-link" type="button">Short Another Link</button>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </div>
    
    <footer>
        <div class="footer-content">
            <p>&copy; 2024 SpidyUrl. Professional URL Shortener Service.</p>
            <div class="footer-links">
                <a href="/about.php">About</a>
                <a href="/contact.php">Contact</a>
                <a href="/privacy.php">Privacy Policy</a>
                <a href="/terms.php">Terms of Service</a>
                <a href="/cookies.php">Cookie Policy</a>
            </div>
        </div>
    </footer>
    
    <script>
        function copyToClipboard() {
            const shortUrlInput = document.getElementById('shortUrl');
            shortUrlInput.select();
            shortUrlInput.setSelectionRange(0, 99999);
            
            try {
                document.execCommand('copy');
                const copyBtn = document.querySelector('.copy-btn');
                const originalText = copyBtn.textContent;
                copyBtn.textContent = 'Copied!';
                copyBtn.style.background = '#10b981';
                
                setTimeout(() => {
                    copyBtn.textContent = originalText;
                    copyBtn.style.background = '';
                }, 2000);
            } catch (err) {
                console.error('Failed to copy text: ', err);
            }
        }
        
        function resetForm() {
            // Clear the URL input field
            const urlInput = document.querySelector('input[name="long_url"]');
            if (urlInput) {
                urlInput.value = '';
                urlInput.focus();
            }
            
            // Hide the result section
            const resultSection = document.querySelector('.result');
            if (resultSection) {
                resultSection.style.display = 'none';
            }
            
            // Hide any messages
            const messageSection = document.querySelector('.message');
            if (messageSection) {
                messageSection.style.display = 'none';
            }
            
            // Smooth scroll to top
            window.scrollTo({
                top: 0,
                behavior: 'smooth'
            });
        }
        
        // Auto-focus on input field
        document.addEventListener('DOMContentLoaded', function() {
            const input = document.querySelector('input[name="long_url"]');
            if (input && !input.value) {
                input.focus();
            }
        });
    </script>
</body>
</html>